/*! firebase-admin v10.3.0 */
"use strict";
/*!
 * @license
 * Copyright 2022 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Channel = exports.Eventarc = void 0;
var validator = require("../utils/validator");
var eventarc_utils_1 = require("./eventarc-utils");
var eventarc_client_internal_1 = require("./eventarc-client-internal");
/**
 * Eventarc service bound to the provided app.
 */
var Eventarc = /** @class */ (function () {
    /**
     * @internal
     */
    function Eventarc(app) {
        if (!validator.isNonNullObject(app) || !('options' in app)) {
            throw new eventarc_utils_1.FirebaseEventarcError('invalid-argument', 'First argument passed to Eventarc() must be a valid Firebase app instance.');
        }
        this.appInternal = app;
    }
    Object.defineProperty(Eventarc.prototype, "app", {
        /**
         * The {@link firebase-admin.app#App} associated with the current Eventarc service
         * instance.
         *
         * @example
         * ```javascript
         * var app = eventarc.app;
         * ```
         */
        get: function () {
            return this.appInternal;
        },
        enumerable: false,
        configurable: true
    });
    Eventarc.prototype.channel = function (nameOrOptions, options) {
        var channel;
        var opts;
        if (validator.isNonEmptyString(nameOrOptions)) {
            channel = nameOrOptions;
        }
        else {
            channel = 'locations/us-central1/channels/firebase';
        }
        if (validator.isNonNullObject(nameOrOptions)) {
            opts = nameOrOptions;
        }
        else {
            opts = options;
        }
        var allowedEventTypes = undefined;
        if (typeof (opts === null || opts === void 0 ? void 0 : opts.allowedEventTypes) === 'string') {
            allowedEventTypes = opts.allowedEventTypes.split(',');
        }
        else if (validator.isArray(opts === null || opts === void 0 ? void 0 : opts.allowedEventTypes)) {
            allowedEventTypes = opts === null || opts === void 0 ? void 0 : opts.allowedEventTypes;
        }
        else if (typeof (opts === null || opts === void 0 ? void 0 : opts.allowedEventTypes) !== 'undefined') {
            throw new eventarc_utils_1.FirebaseEventarcError('invalid-argument', 'AllowedEventTypes must be either an array of strings or a comma separated string.');
        }
        return new Channel(this, channel, allowedEventTypes);
    };
    return Eventarc;
}());
exports.Eventarc = Eventarc;
/**
 * Eventarc Channel.
 */
var Channel = /** @class */ (function () {
    /**
     * @internal
     */
    function Channel(eventarc, name, allowedEventTypes) {
        if (!validator.isNonNullObject(eventarc)) {
            throw new eventarc_utils_1.FirebaseEventarcError('invalid-argument', 'First argument passed to Channel() must be a valid Eventarc service instance.');
        }
        if (!validator.isNonEmptyString(name)) {
            throw new eventarc_utils_1.FirebaseEventarcError('invalid-argument', 'name is required.');
        }
        this.nameInternal = name;
        this.eventarcInternal = eventarc;
        this.allowedEventTypes = allowedEventTypes;
        this.client = new eventarc_client_internal_1.EventarcApiClient(eventarc.app, this);
    }
    Object.defineProperty(Channel.prototype, "eventarc", {
        /**
         * The {@link firebase-admin.eventarc#Eventarc} service instance associated with the current `Channel`.
         *
         * @example
         * ```javascript
         * var app = channel.eventarc;
         * ```
         */
        get: function () {
            return this.eventarcInternal;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(Channel.prototype, "name", {
        /**
         * The channel name as provided during channel creation. If it was not specifed, the default channel name is returned
         * ('locations/us-central1/channels/firebase').
         */
        get: function () {
            return this.nameInternal;
        },
        enumerable: false,
        configurable: true
    });
    /**
     * Publishes provided events to this channel. If channel was created with `allowedEventTypes` and event type is not
     * on that list, the event is ignored.
     *
     * @param events - CloudEvent to publish to the channel.
     */
    Channel.prototype.publish = function (events) {
        return this.client.publish(events);
    };
    return Channel;
}());
exports.Channel = Channel;
